<?php

/**
 * This is the model class for table "course". 
 *
 * The followings are the available columns in table 'course':
 * @property string $id
 * @property string $name
 * @property string $short_description
 * @property string $description
 * @property string $cost
 * @property string $instruction_methods
 * @property string $evaluation_method
 *
 * The followings are the available model relations:
 * @property CourseHasCategory[] $courseHasCategories
 * @property Materials[] $materials
 * @property UserCourseAccess[] $userCourseAccesses
 */
class Course extends CActiveRecord
{

	public $course_categories; 

	/**
	 * Returns the static model of the specified AR class.
	 * @param string $className active record class name.
	 * @return Course the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'course';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		// NOTE: you should only define rules for those attributes that
		// will receive user inputs.
		return array(
			array('name', 'required'),
			array('name, cost, ceu_credit, promo_code, passing_percentage, required_within_time, staff_requirement, frequency_required', 'length', 'max'=>60),
			array('short_description, description, in_person, essay_question, essay_question_content, instruction_methods, evaluation_method, video', 'safe'),
			array('powerpoint, video', 'file', 'allowEmpty'=>true),
			// The following rule is used by search().
			// Please remove those attributes that should not be searched.
			array('id, name, short_description, description, cost, instruction_methods, evaluation_method, courseHasCategories', 'safe', 'on'=>'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		// NOTE: you may need to adjust the relation name and the related
		// class name for the relations automatically generated below.
		return array(
			'courseHasCategories' => array(self::HAS_MANY, 'CourseHasCategory', 'course_id'),
			'materials' => array(self::HAS_MANY, 'Materials', 'course_id'),
			'userCourseAccesses' => array(self::HAS_MANY, 'UserCourseAccess', 'course_id'),
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id' => 'ID',
			'name' => 'Name',
			'short_description' => 'Short Description',
			'description' => 'Description',
			'cost' => 'Cost',
			'essay_question' => 'Include Essay Question?',
			'essay_question_content' => 'Essay Question',
			'promo_code' => 'Promo code for free access',
			'instruction_methods' => 'Instruction Methods',
			'evaluation_method' => 'Evaluation Method',
			'course_categories' => 'Assign Categories',
			'passing_percentage' => 'Passing Percentage',
			'required_within_time' => 'Required within hire date'
		);
	}


	public function isRequired() {

		$user = User::model()->findByPk(Yii::app()->user->id);

		foreach ($user->staff_type as $staff_type) {
			
			foreach ($this->courseHasCategories as $category) {

				// echo $category->courseCategory->id.'='.$staff_type.'<br>';

				if($category->courseCategory->id == $staff_type)
					return true;
			}

		}

		return false;
	}


	public static function getOfferingFromCategory($cat)
	{
		$cmd = Yii::app()->db->createCommand();
		$cmd->select('offering_id');
		$cmd->from('offering_has_category');
		$cmd->where('course_category_id='.$cat);
		$offering_id = $cmd->queryScalar();
		return $offering_id;
	}

	public static function getCourseNameByID($id)
	{
		$cmd = Yii::app()->db->createCommand();
		$cmd->select('name');
		$cmd->from('course');
		$cmd->where('id='.$id);
		$name = $cmd->queryScalar();
		return $name;
	}

	public static function paymentActivated () {

		$cmd = Yii::app()->db->createCommand();
		$cmd->select('payment_activated');
		$cmd->from('settings');
		$payment_activated = $cmd->queryScalar(); 

		return $payment_activated;
	}


	public static function getPassingPercentage($id) {

		$cmd = Yii::app()->db->createCommand();
		$cmd->select('passing_percentage');
		$cmd->from('course');
		$cmd->where('id='.$id);
		$percentage = $cmd->queryScalar(); 

		if(!$percentage)
			$percentage = 70;

		return $percentage;
	}

	public function getCertificateFilename($user_id){
    	
    	$user=User::model()->findByPk($user_id);

    	// Find filename for certificate
    	$filename = $user->fname.' '.$user->lname;
    	$filename .= '_'.$this->name;
    	$filename = preg_replace('/[^a-zA-Z0-9_.-]/', '_', $filename);

    	return $filename;
    }



	public static function getCourseIds(){
        // Get access code
		$cmd = Yii::app()->db->createCommand();
		$cmd->select('id');
		$cmd->from('course');
		$user = $cmd->queryAll(); 
		return $user;
    }



	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
	 */
	public function search()
	{
		// Warning: Please modify the following code to remove attributes that
		// should not be searched.

		$criteria=new CDbCriteria;
	
		$criteria->compare('id',$this->id,true);
		$criteria->compare('name',$this->name,true);
		$criteria->compare('short_description',$this->short_description,true);
		$criteria->compare('description',$this->description,true);
		$criteria->compare('cost',$this->cost,true);
		$criteria->compare('instruction_methods',$this->instruction_methods,true);
		$criteria->compare('evaluation_method',$this->evaluation_method,true);


		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
			'pagination' => array('pageSize' => 25),
		));
	}





	public function generateCertificate() {

		// Get completion date
		$cmd = Yii::app()->db->createCommand();
		$cmd->select('time_completed');
		$cmd->from('user_course_access');
		$cmd->where('user_id='.Yii::app()->user->id);
		$cmd->andWhere('course_id='.$this->id);
		$time_completed = $cmd->queryScalar(); 
		$time_completed = date('F j, Y', $time_completed);
		

		require_once(Yii::app()->basePath . '/extensions/fpdi/tcpdf/tcpdf.php'); 
		require_once(Yii::app()->basePath . '/extensions/fpdi/fpdi.php'); 
		

		$baseUrl = Yii::app()->baseUrl; 

		// initiate FPDI
		$pdf = new FPDI();

		$pdf->setPrintHeader(false);
		$pdf->setPrintFooter(false);

		// add a page
		$pdf->AddPage('L');

		// set the source file
		$pdf->setSourceFile("course_assets/blank-certificate.pdf");
		// import page 1
		$tplIdx = $pdf->importPage(1);
		// use the imported page and place it at point 10,10 with a width of 100 mm
		$pdf->useTemplate($tplIdx, 18, 0, 252);


		$pdf->SetTitle('TCPDF Example 001');

		// Add Name
		$pdf->SetFont('Helvetica','',18);
		$pdf->SetTextColor(0, 0, 0);
		$pdf->SetXY(0, 95);
		$pdf->Cell(0, 0, Yii::app()->user->fname.' '.Yii::app()->user->lname, 0, 0, 'C');


		// Add Course Title and CEU
		$pdf->SetFont('Helvetica','',18);
		$pdf->SetTextColor(0, 0, 0);
		$pdf->SetXY(0, 115);
		$pdf->Cell(0, 0, $this->name.' ('.$this->ceu_credit.' hours)', 0, 0, 'C');

		// Add Date
		$pdf->SetFont('Helvetica','',14);
		$pdf->SetTextColor(0, 0, 0);
		$pdf->SetXY(0, 123);
		$pdf->Cell(0, 0, $time_completed, 0, 0, 'C');


		// Add CEU
		$pdf->SetFont('Helvetica','',11);
		$pdf->SetTextColor(0, 0, 0);
		$pdf->SetXY(149.5, 132.6);
		$pdf->Cell(0, 0, $this->ceu_credit, 0, 0, 'C');

		$filename = Yii::app()->user->fname.' '.Yii::app()->user->lname;
		$filename .= '_'.$this->name;
		$filename = preg_replace('/[^a-zA-Z0-9_.-]/', '_', $filename);

		$save_url = 'course_assets/certificates/'.$filename.'.pdf';

		$pdf->Output($save_url, 'F');

		return $save_url;
	}



	public function getEssayStatus() {
		
		// Get attempts
		$cmd = Yii::app()->db->createCommand();
		$cmd->select('essay_grade');
		$cmd->from('user_course_access');
		$cmd->where('user_id='.Yii::app()->user->id);
		$cmd->andWhere('course_id='.$this->id);
		$result = $cmd->queryRow(); 

		return $result['essay_grade'];
	}


}