<?php

/**
 * This is the model class for table "user".
 *
 * The followings are the available columns in table 'user':
 * @property string $id
 * @property string $fname
 * @property string $lname
 * @property string $pass
 * @property string $passOld
 * @property string $passNew
 * @property string $passNewCompare
 * @property string $sex
 * @property string $email
 * @property string $email_compare
 * @property string $birthdate_day
 * @property string $birthdate_month
 * @property string $birthdate_year
 * @property string $join_date
 * @property string $access_code
 */
class User extends CActiveRecord
{

	public $email_compare; 
	public $birthdate_day; 
	public $birthdate_month; 
	public $birthdate_year; 
	public $passOld; 
	public $passNew; 
	public $passNewCompare; 
	public $resetPassNew;
	public $resetPassNewCompare;
	public $passUpdate;
	public $profileImage;
	public $linkedAccount;
	public $confirmation_code;
	public $access_code;
	public $CEU_count;
	public $access_level;
	public $country;
	public $staff_type;
	public $hired_date;


	/**
	 * Returns the static model of the specified AR class.
	 * @param string $className active record class name.
	 * @return User the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'user';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		// NOTE: you should only define rules for those attributes that
		// will receive user inputs.
		return array(
			//Step 1 Scenario
			
			array('fname, lname, email, pass', 'required'),
			array('fname, lname, email, birthdate_day, birthdate_month birthdate_year, access_code, staff_type', 'length', 'max'=>60),
			array('access_level', 'default', 'value'=>0),
			array('pass', 'length', 'max'=>64),

			//Change Password
			array('passOld', 'changePassword'),
			array('passNew', 'length', 'max'=>64),
			array('passNewCompare', 'length', 'max'=>64),
			array('passNew', 'compare', 'compareAttribute'=>'passNewCompare', 'message'=>'Password doesn\'t match'),
			array('passUpdate', 'length', 'max'=>64),

			//Password Reset
			array('resetPassNew', 'length', 'max'=>64),
			array('resetPassNewCompare', 'length', 'max'=>64),
			array('resetPassNewCompare', 'compare', 'compareAttribute'=>'resetPassNew', 'message'=>'Password doesn\'t match'),

			array('email', 'email'),
			array('email', 'unique'),
			array('email_compare', 'compare', 'compareAttribute'=>'email', 'message'=>'Email doesn\'t match', 'on'=>'create'),

			array('birthdate_day, active, birthdate_year, hired_date', 'numerical', 'integerOnly'=>true),
			array('join_date', 'default', 'value'=>new CDbExpression('NOW()'),
				'on'=>'insert'),


			// The following rule is used by search().
			// Please remove those attributes that should not be searched.
			
			array('id, fname, lname, pass, email, join_date', 'safe', 'on'=>'search'),
		);
	}

	public function changePassword($attribute,$params) {
	    if($this->passOld || $this->passNew || $this->passNewCompare) {
			if($this->pass != hash_hmac('sha256',$this->passOld,Yii::app()->params['encryptionKey'])) 
				$this->addError('passOld','Incorrect password.');
	    }
	}

	public function radioRequired($attribute,$params) {
	    if($this->$attribute == '0') 
	    	$this->addError($attribute,'Please select one of the options'.$this->$attribute);
	}


	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		// NOTE: you may need to adjust the relation name and the related
		// class name for the relations automatically generated below.
		return array(
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{

		$labels = array(
			'id' => 'ID',
			'fname' => 'First Name',
			'lname' => 'Last Name',
			'pass' => 'Password',
			'passOld' => 'Current Password',
			'passNew' => 'New Password',
			'passNewCompare' => 'Confirm Password',
			'email' => 'Email Address',
			'email_compare' => 'Confirm Email Address',
			'country' => 'Country',
			'join_date' => 'Date Entered',
			'linkedAccount' => 'Linked Account Number'
		);

		if($this->scenario == 'settings') {
			$labels['pass'] = 'Current Password';
		}

		return $labels;

	}

	protected function beforeSave() {

		//Hash Password
		if ($this->isNewRecord) {
			$this->pass = hash_hmac('sha256', $this->pass, Yii::app()->params['encryptionKey']); 
		}

		//Save code for email confirmation
		if ($this->isNewRecord) {
			$this->confirmation_code = mt_rand(10000000, 99999999);
		}    

		//Set CEU Count at zero
		if ($this->isNewRecord) {
			$this->CEU_count = 0;
		}	

    	//Change Password on settings page
    	if($this->passNew)
    		$this->pass = hash_hmac('sha256', $this->passNew, Yii::app()->params['encryptionKey']);

    	//Reset Password on reset page
    	if($this->resetPassNew)
    		$this->pass = hash_hmac('sha256', $this->resetPassNew, Yii::app()->params['encryptionKey']);

    	//Reset Password on admin update
    	if($this->passUpdate)
    		$this->pass = hash_hmac('sha256', $this->passUpdate, Yii::app()->params['encryptionKey']);
                
        

    	
    	return parent::beforeSave();
	}


	protected function beforeValidate() {

		//Check if access code is set, if so make sure they match
		if(!$this->resetPassNew && $this->scenario != 'update') {

	        if($this->getAccessCode()) {
	        	if($this->getAccessCode() != $this->access_code) 
	        		$this->addError('access_code','Incorrect Access Code.');
	        }
	    }


	    //Format Hired date
		if ($this->scenario == 'update' && $this->hired_date) {
				$this->hired_date = strtotime($this->hired_date);
		} else if ($this->hired_date) {
				$this->hired_date = strtotime($this->hired_date);
		}




		return parent::beforeValidate();
	}


	protected function afterFind() {

		// Format Hired date for display
		if($this->hired_date) {
				$this->hired_date = date('m/d/Y', $this->hired_date);
		}

		// Convert staff type to array
		if($this->staff_type) { 
			$this->staff_type = unserialize($this->staff_type);
		} else {
			$this->staff_type = array();
		}


		return parent::afterFind();
	}


	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
	 */
	public function search()
	{
		// Warning: Please modify the following code to remove attributes that
		// should not be searched.

		$criteria=new CDbCriteria;

		// $criteria->compare('access_level',0);
		$criteria->compare('active',$this->active,true);
		$criteria->compare('id',$this->id,true);
		$criteria->compare('fname',$this->fname,true);
		$criteria->compare('lname',$this->lname,true);
		$criteria->compare('pass',$this->pass,true);
		$criteria->compare('email',$this->email,true);
		$criteria->compare('join_date',$this->join_date,true);

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
			'pagination' => array('pageSize' => 55),

		));
	}


	public static function getAccessCode(){
        // Get access code
		$cmd = Yii::app()->db->createCommand();
		$cmd->select('access_code');
		$cmd->from('settings');
		$access_code = $cmd->queryScalar(); 
		return $access_code;
    }

    public function getStaffTypes(){
        // Get staff types
		$cmd = Yii::app()->db->createCommand();
		$cmd->select('staff_type_select');
		$cmd->from('settings');
		$staff_types = $cmd->queryScalar(); 
		if($staff_types) {
			$staff_types = explode(",",$staff_types);
			$staff_types = array_combine($staff_types, $staff_types);
			return $staff_types;
		} else {
			return NULL;
		}
    }


    public static function getUserById($id){
        // Get access code
		$cmd = Yii::app()->db->createCommand();
		$cmd->select('*');
		$cmd->from('user');
		$cmd->where('id='.$id);
		$user = $cmd->queryRow(); 
		return $user;
    }



    public static function checkFullCompliance ($user_id) {

    	// Checks all courses to see if the student is completely compliant

    	//Get all courses
		$courseIds = Course::getCourseIds();
		$compliant = 1;

    	foreach ($courseIds as $course) {
    		
    		$compliance = User::checkCompliance($user_id, $course['id']);
    		if($compliance['compliant'] == 0)
    			$compliant = 0;
    	}

    	if($compliant)
    		return true;

    }



    public static function checkCompliance($user_id, $course_id){
        // Get user data
		$cmd = Yii::app()->db->createCommand();
		$cmd->select('*')->from('user')->where('id='.$user_id);
		$user = $cmd->queryRow(); 

		//Get course data
		$cmd = Yii::app()->db->createCommand();
		$cmd->select('*')->from('course')->where('id='.$course_id);
		$course = $cmd->queryRow(); 

		// Get existing course completions
		$results = UserCourseAccess::checkStatus($course_id, $user_id);


		// Calculate compliance
		$current_date = time();
		$required_by_date = $user['hired_date'] + ($course['required_within_time'] * 86400);
		
		if(isset($results) && $results['status'] == 'Completed') {
			$expires_time = $results['time_completed'] + ($course['frequency_required'] * 86400);
		}


		// By default they should be compliant
		$compliance = array();
		$compliance['compliant'] = 1;


		// Check users category to find out if the course is required
		$course_categories = CourseCategory::getCourseCategoryId($course_id);
		$staff_types = unserialize($user['staff_type']);

		
			


		


		// Return data
		return $compliance;

    }



}